<?php
/**
 * Smart Brands for WooCommerce
 *
 * @package           Smart Brands for WooCommerce
 * @author            ShapedPlugin
 * @license           GPL-2.0-or-later
 *
 * Plugin Name:       Smart Brands for WooCommerce
 * Plugin URI:        https://shapedplugin.com/smart-brands/
 * Description:       Smart Brands for WooCommerce plugin allows you to add brands for products and gain credibility by displaying them in your WooCommerce shop.
 * Version:           2.0.2
 * Requires at least: 5.2
 * Requires PHP:      7.2
 * WC tested up to:   10.0.4
 * Requires Plugins:  woocommerce
 * Author:            ShapedPlugin LLC
 * Author URI:        https://shapedplugin.com/
 * Text Domain:       smart-brands-for-woocommerce
 * License:           GPL v2 or later
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 */

// don't call the file directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class_exists( 'ShapedPlugin\SmartBrands\SmartBrands' ) || require_once __DIR__ . '/vendor/autoload.php';

use ShapedPlugin\SmartBrands\SmartBrands;

define( 'SMART_BRANDS_VERSION', '2.0.2' );
define( 'SMART_BRANDS_FILE', __FILE__ );
define( 'SMART_BRANDS_PATH', dirname( SMART_BRANDS_FILE ) );

// Declare that the plugin is compatible with WooCommerce High-Performance order storage feature.
add_action(
	'before_woocommerce_init',
	function () {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
		}
	}
);

if ( ! function_exists( 'smart_brands_get_brand_taxonomy' ) ) {
	/**
	 * Retrieve the taxonomy slug used for brands.
	 *
	 * This function returns either 'sp_smart_brand' or 'product_brand' depending on the
	 * WooCommerce version. Starting from WooCommerce 9.6, it uses the 'product_brand'
	 * taxonomy, otherwise it uses the 'sp_smart_brand' taxonomy.
	 *
	 * @return string The taxonomy slug used for brands.
	 */
	function smart_brands_get_brand_taxonomy() {
		if ( ! function_exists( 'is_plugin_active' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$woo_version = get_option( 'woocommerce_version', '0' );
		return ( is_plugin_active( 'woocommerce/woocommerce.php' ) && version_compare( $woo_version, '9.6', '>=' ) ) ? 'product_brand' : 'sp_smart_brand';
	}
}
update_option( 'smart_brands_brand_taxonomy', smart_brands_get_brand_taxonomy() );

if ( ! function_exists( 'smart_brand_for_wc' ) ) {
	/**
	 * Shortcode converter function
	 *
	 * @param  int $id shortcode id.
	 * @return void
	 */
	function smart_brand_for_wc( $id ) {
		echo do_shortcode( '[smart_brand_for_wc id="' . $id . '"]' );
	}
}

/**
 * Init the SmartBrands plugin.
 *
 * @since 1.0.0
 */
function launch_smart_brands() {
	// Launch the plugin.
	$smart_brand_for_wc = new SmartBrands();
	$smart_brand_for_wc->run();
}

/**
 * Checks if the Smart Brands Pro plugin is active.
 *
 * This function verifies whether the Pro version of the Smart Brands plugin
 * is currently activated in the WordPress environment.
 *
 * @return bool True if Smart Brands Pro is active, false otherwise.
 */
function is_smart_brands_pro_active() {
	include_once ABSPATH . 'wp-admin/includes/plugin.php';
	if ( ( is_plugin_active( 'smart-brands-pro/smart-brands-pro.php' ) || is_plugin_active_for_network( 'smart-brands-pro/smart-brands-pro.php' ) ) ) {
		return true;
	}
}

// Launch the plugin only if the Smart Brands Pro plugin is not active.
if ( ! is_smart_brands_pro_active() ) {
	// Launch the plugin.
	launch_smart_brands();
}
