<?php
/**
 * Class for creating custom meta fields for page and post.
 *
 * @package Orchid_Store
 */

if ( ! class_exists( 'Orchid_Store_Custom_Fields' ) ) {
	/**
	 * Widget class - Orchid_Store_Custom_Fields.
	 *
	 * @since 1.0.0
	 *
	 * @package orchit_store
	 */
	class Orchid_Store_Custom_Fields {
		/**
		 * Define id, name and description of the widget.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {

			// Register post meta fields and save meta fields values.
			add_action( 'admin_init', array( $this, 'register_post_meta' ) );
			add_action( 'save_post', array( $this, 'save_sidebar_position_meta' ) );
		}

		/**
		 * Register post custom meta fields.
		 *
		 * @since    1.0.0
		 */
		public function register_post_meta() {

			add_meta_box(
				'sidebar_position_metabox',
				esc_html__( 'Sidebar Position', 'orchid-store' ),
				array( $this, 'sidebar_position_meta' ),
				array( 'post', 'page' ),
				'side',
				'default'
			);
		}

		/**
		 * Custom Sidebar Post Meta.
		 *
		 * @since    1.0.0
		 */
		public function sidebar_position_meta() {

			global $post;

			$sidebar_position_key = 'orchid_store_sidebar_position';

			$sidebar_position = get_post_meta( $post->ID, $sidebar_position_key, true );

			if ( empty( $sidebar_position ) ) {
				$sidebar_position = 'right';
			}

			wp_nonce_field( 'sidebar_position_meta_nonce', 'sidebar_position_meta_nonce_id' );

			$choices = array(
				'right' => esc_html__( 'Right', 'orchid-store' ),
				'left'  => esc_html__( 'Left', 'orchid-store' ),
				'none'  => esc_html__( 'Fullwidth', 'orchid-store' ),
			);
			?>
			<table width="100%" border="0" class="options" cellspacing="5" cellpadding="5">
				<tr>
					<td>
						<select class="widefat" name="sidebar_position" id="sidebar_position">
							<?php
							foreach ( $choices as $key => $choice ) {
								?>
								<option
									value="<?php echo esc_attr( $key ); ?>"
									<?php selected( $key, $sidebar_position ); ?>
								>
									<?php echo esc_html( $choice ); ?>
								</option>
								<?php
							}
							?>
						</select>
					</td>
				</tr> 
			</table>   
			<?php
		}

		/**
		 * Save Custom Sidebar Position Post Meta.
		 *
		 * @since    1.0.0
		 */
		public function save_sidebar_position_meta() {

			global $post;

			// Bail if we're doing an auto save.
			if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
				return;
			}

			// if our nonce isn't there, or we can't verify it, bail.
			if ( ! isset( $_POST['sidebar_position_meta_nonce_id'] ) || ! wp_verify_nonce( sanitize_key( $_POST['sidebar_position_meta_nonce_id'] ), 'sidebar_position_meta_nonce' ) ) {
				return;
			}

			// if our current user can't edit this post, bail.
			if ( ! current_user_can( 'edit_post', $post->ID ) ) {
				return;
			}

			if ( isset( $_POST['sidebar_position'] ) ) {

				update_post_meta( $post->ID, 'orchid_store_sidebar_position', sanitize_text_field( wp_unslash( $_POST['sidebar_position'] ) ) );
			}
		}
	}
}

$orchid_store_custom_fields = new Orchid_Store_Custom_Fields();
